//Micromite Hardware (V3 Backpack)
//3.5in LCD (ILI9488)
//15 (RB6) = TIRQ
//7 (RB3) = TCS
//2 (RA0) = LCD DC
//23 (RB12) = LCD RST
//6 (RB2)= LCD CS
//3 (RA1) = SPI MOSI
//14 (RB5) = SPI MISO
//25 (RB14) = SPI SCK
//26 (RB15) =  LCD backlight
//4 (RB0) = SD CS

//board with audio on 5/24, data on 9
//IOs:
//5 (RB1)= right audio PWM
//9 (RA2)= DATA for DLC
//24 (RB13) = left audio PWM

#include <xc.h>

#include "config.h"
#include "backpack.h"
#include "font_arial.h"
#include "small_font.h"
#include "sclogo.h"
#include "dlc_image.h"
#include "dlc.h"
#include "sdcard/spi.h"
#include "sdcard/sd.h"
#include "sdcard/ff.h"
#include "uart.h"
#include "WAV.h"

#define _XTAL_FREQ 40000000

extern unsigned char DLC_data[65];       //array of data to be output
extern int DLC_state;           //0=off, otherwise value is machine state
extern char DCL_run;            //boolean state
extern char DLC_seq_state;      //are we processing a sequence?
extern char WAV_state;          //0=off, 1=playing
extern unsigned long tmr;       //timer counter incremented by DLC ISR
extern unsigned long seqtmr;    //timer counter incremented by events in the sequence
long seqtime=0;                 //total time

extern char wavBuffer[BUFFERCOUNT][BUFFERSIZE];              //heaps of RAM available
extern char seqBuffer[BUFFERCOUNT][BUFFERSIZE];                
extern int wavBufferBytes[BUFFERCOUNT];                      //number of bytes available to play
extern int seqBufferBytes[BUFFERCOUNT];
extern char WAV_state;                                       //0=off, 1=playing

int seqPage=0;
int wavPage=0;

char loopMode=0;                //loop playback flag
char seqName[256]="";           //for UI

unsigned char memoryCardSystemUp=0;              // flag indicates whether SD card has been detected & configured
static FATFS fs;
uint16_t stat;
uint16_t d=0,i=0;
char rootbuf[5] = "0:\\";
const char* pbuf;    
FATFS* pfs;
DIR dd;
FILINFO fno;        //generic file info
int fresult;        //to keep any file result
FIL wavfile;    //for playing wav file
FIL seqfile;    //for playing sequence file    
int fcount=0;       //number of root files

//FRESULT f[BUF_COUNT];
//FIL filetoopen[BUF_COUNT];
struct backpackbutton scan;
struct backpackbutton test;
struct backpackbutton play;
struct backpackbutton stop;
struct backpackbutton play_loop;
struct backpackbutton next_button;
struct backpackbutton prev_button;
struct backpackbutton tone_button;

const char text_scan[]="Scanning";
const char text_reset[]="Reset";
const char text_continue[]="Continue";
const char text_test[]="Test mode";
const char text_exit[]="Exit";
const char text_toggle[]="Toggle";
//const char text_all[]="Play all";
//const char text_one[]="Play one";
const char text_loop[]="Loop";
const char text_next[]="Next";
const char text_prev[]="Previous";
const char text_stop[]="Pause";
const char text_play[]="Play";
const char text_tone[]="Tone";
const char text_loop_off[]="Loop off";
const char text_loop_one[]="Loop one";
const char text_loop_all[]="Loop all";

//helpers
void mainScreen(void);
int checkFiles(void);
void SoftReset(void);
void testmode(void);
void playmode(void);
void showMessage(int i);     //display error message/file state
int getFile(int);
int checkifLS(char* s);
int scopy(char* src, char* dst, int n);
void dispName(int y, char* s);
int setWavName(char* w, char*s);
void fillWavBuffer(void);    
char checkWavBuffer(void);          //load buffers, return 0 on error
char checkSeqBuffer(void);
int getWavInfo(void);               //get wav attributes
void showWavInfo(void);             //display on LCD
void buttonRelease(struct backpackbutton *b);
void getSeqtime();                  //scan file and add up times
void showTimes(void);
void clearBuffers(void);            //clear all buffers and output state

void main(void) {
    int i;
    uart_init(38400);
    uart_print("Starting OK\r\n");
    backpackinit();
    backpacksetbuttonobject(&scan,280,260,440,310,text_scan);
    mainScreen();
    delay(500);
    //start up and run DLC data always
    dlc_init();
    DLC_state=1;
    DLC_run=1;    
    i=checkFiles();
    uart_printi(i);
    fcount=i;       //save
    uart_print(" files found\r\n");            
    SPI1BRG=1;  //SD leaves it slow after error
    showMessage(fcount);
    if(i>0){
        scan.text=text_continue;
    }else{
        scan.text=text_reset;
    }
    buttonStyle(&scan,0);    
    buttonStyle(&test,0);    
    uart_printi(SPI1BRG);
    uart_print("=BRG\r\n");            

    while(1){
        if(backpackbuttonobjecttouched(&scan)){
            buttonRelease(&scan);
            if(i>0){
                playmode();
                mainScreen();
                showMessage(fcount);
                buttonStyle(&test,0);                
            }else{
                SoftReset();                
            }
        }
        if(backpackbuttonobjecttouched(&test)){
            buttonRelease(&test);
            testmode();
            mainScreen();
            showMessage(fcount);
            buttonStyle(&test,0);                
        }
    }
    return;
}

void mainScreen(void){
    backpackrotate(4);
    backpackclear(BLACK);
    //no text:
    //backpackmonobitmapscale(70,0,sclogo,2);
    //backpackmonobitmapscale(115,128,dlc_logo,1);
    
    //small logos and text
    backpackmonobitmapscale(155,0,sclogo,1);
    backpackchararrayfont(112,66,"Flexible Digital",GREY,BLACK,ARIAL);
    backpackchararrayfont(88,96,"Lighting Controller",GREY,BLACK,ARIAL);
    backpackmonobitmapscale(115,128,dlc_logo,1);
    
    //main logo and text
    //backpackmonobitmapscale(70,0,sclogo,2);
    //backpackchararrayfont(112,130,"Flexible Digital",GREY,BLACK,ARIAL);
    //backpackchararrayfont(88,160,"Lighting Controller",GREY,BLACK,ARIAL);
    
    buttonStyle(&scan,0);    
    backpacksetbuttonobject(&test,40,260,200,310,text_test);
}

int checkFiles(void){
    int i;
    memoryCardSystemUp=1;                       //card assumed in place
    pbuf=rootbuf;
    for(i=0;i<10;i++){
        fresult=f_mount(0, &fs);          //mount fs
        uart_printi(fresult);
        uart_print("=mount\r\n");                
        stat=auto_mount(&pbuf, &pfs, 0);        //inits fs and spi etc
        uart_printi(stat);
        uart_print("=stat\r\n");        
        if(stat==FR_OK){break;}
    }
    if(stat==FR_OK){
        fresult=f_opendir(&dd,rootbuf);
        uart_printi(fresult);
        uart_print("=opendir\r\n");        
        i=0;
        while(i<255){
        //for(i=0;i<255;i++){
            fresult=f_readdir(&dd,&fno);
            uart_printi(fresult);
            uart_print("=f_readdir\r\n");   
            if((fresult!=FR_OK)||(fno.fname[0]==0)){        //fail or empty name
                break;
            }else{
                if(((fno.fattrib&AM_DIR)==0)&&checkifLS(fno.fname)){
                    uart_printi(i);
                    uart_print(" : ");
                    uart_print(fno.fname);
                    uart_print(" : ");
                    uart_print(fno.lfname);
                    uart_print("\r\n");
                    i++;
                }
            }
        }
        return i;           //files found
    }else{
        return -stat;       //error
    }    
}

void __attribute__((noreturn)) SoftReset(void){     //reboots
    __builtin_enable_interrupts();    
    SYSKEY = 0;                 // force lock
    SYSKEY = 0xAA996655;        // unlock sequence
    SYSKEY = 0x556699AA;        //  
    RSWRSTSET = 1;              //arm reset
    unsigned int dummy;
    dummy = RSWRST;             //dummy read to execute
    while(1){}
}

void testmode(void){
    backpackclear(BLACK);
    backpackchararrayfont(168,0,"TEST MODE",BUTTON_OTHER,BUTTON_BACK,ARIAL);
    const char* sOld;
    int cBase=0;
    char drawAl1=1;
    int i;
    char tonesOn=0;
    sOld=scan.text;
    backpacksetbuttonobject(&scan,340,260,460,310,text_exit);
    backpacksetbuttonobject(&test,20,260,140,310,text_toggle);    
    backpacksetbuttonobject(&tone_button,180,260,300,310,text_tone);
    buttonStyle(&scan,0);    
    buttonStyle(&test,0);    
    buttonStyle(&tone_button,tonesOn);    
    while(1){
        if(drawAl1){
            for(i=0;i<16;i++){
                backpacknint(7+i*30, 40, cBase+i,2, BUTTON_FORE,BUTTON_BACK,myke_font);
                backpacknint(4+i*30, 240, DLC_data[cBase+i+1],3, BUTTON_FORE,BUTTON_BACK,myke_font);
                backpackbox(i*30+5,60,i*30+25,188,BUTTON_BACK);  //erase
                backpackhline(i*30+5,60,i*30+25,BUTTON_OTHER);
                backpackhline(i*30+5,230,i*30+25,BUTTON_OTHER);
                backpackvline(i*30+15,60,230,BUTTON_OTHER);
                backpackhline(i*30+5,230-((DLC_data[cBase+i+1]*2)/3),i*30+25,BUTTON_FORE);//draw new                
            }
        }
        drawAl1=0;
        int x,y;
        x=backpacktouchx();
        y=backpacktouchy();
        if((y>40)&&(y<240)){        //in slider zone
            if(((x+25)%30)<20){     //close to slider
                x=(x+5)/30;
                y=230-y;
                y=(y*3)/2;
                if(y<0){y=0;}
                if(y>255){y=255;}
                backpackhline(x*30+5,230-((DLC_data[cBase+x+1]*2)/3),x*30+25,BUTTON_BACK);//erase old
                backpackvline(x*30+15,60,230,BUTTON_OTHER);                             //draw vertical
                backpackhline(x*30+5,60,x*30+25,BUTTON_OTHER);
                backpackhline(x*30+5,230,x*30+25,BUTTON_OTHER);
                DLC_data[cBase+x+1]=y;
                backpackhline(x*30+5,230-((DLC_data[cBase+x+1]*2)/3),x*30+25,BUTTON_FORE);//draw new                
                backpacknint(4+x*30, 240, DLC_data[cBase+x+1],3, BUTTON_FORE,BUTTON_BACK,myke_font);
            }
        }
        
        if(backpackbuttonobjecttouched(&test)){
            buttonRelease(&test);
            cBase=(cBase+16)&48;
            drawAl1=1;
        }                
        if(backpackbuttonobjecttouched(&scan)){
            buttonRelease(&scan);
            break;
        }       
        if(backpackbuttonobjecttouched(&tone_button)){
            if(tonesOn){tonesOn=0;}else{tonesOn=1;}
            buttonStyle(&tone_button,tonesOn);    
            buttonRelease(&tone_button);
            buttonStyle(&tone_button,tonesOn);    
        }                
        //output a test tone
        if(tonesOn){
            if(WAV_state == 0){
                int p,q;
                for(p=0;p<BUFFERCOUNT;p++){
                    for(q=0;q<BUFFERSIZE;q++){
                        wavBuffer[p][q]=sin64[q&63];                    
                    }
                    wavBufferBytes[p]=BUFFERSIZE;
                }
                wav_bytes_per_sample=1;
                wav_signed=0;
                wav_sample_rate=38400;
                wav_left_offset=0;             
                wav_right_offset=0;            
                WAV_state=1;
            }else{
                int p,q;
                for(p=0;p<BUFFERCOUNT;p++){
                    if(wavBufferBytes[p]==0){
                        for(q=0;q<BUFFERSIZE;q++){
                            wavBuffer[p][q]=sin64[q&63];                    
                        }
                        wavBufferBytes[p]=BUFFERSIZE;                    
                    }
                }            
            }
        }        
    }
    for(i=1;i<65;i++){DLC_data[i]=0;}       //turn all off
    //restore buttons
    backpacksetbuttonobject(&scan,280,260,440,310,sOld);
    backpacksetbuttonobject(&test,40,260,200,310,text_test);    
}

void playmode(void){
    long lasttmr, lastseqtime;       //update display on change
    char pState=0;      //stopped, 1=paused, 2=playing
    char pauseButtonState=0;    //to hold button state
    long mstimer=0;     
    int n;              //used for file reads, get #bytes received
    int j;
    int r;
    int fptr=0;           //index into number of files (0-i)
    r=getFile(fptr);      //init
    backpackclear(BLACK);
    backpackchararrayfont(20,5,"PLAY MODE",BUTTON_OTHER,BUTTON_BACK,ARIAL);
    const char* sOld;
    sOld=scan.text;
    scan.text=text_exit;
    loopMode=0;                 //to display correctly
    buttonStyle(&scan,0);    
    backpacksetbuttonobject(&play,40,200,200,250,text_play);
    backpacksetbuttonobject(&stop,280,200,440,250,text_stop);
    backpacksetbuttonobject(&play_loop,40,260,200,310,text_loop_off);
    backpacksetbuttonobject(&next_button,280,140,440,190,text_next);
    backpacksetbuttonobject(&prev_button,40,140,200,190,text_prev);    
    buttonStyle(&play,0);    
    buttonStyle(&stop,0);    
    buttonStyle(&play_loop,0);    
    buttonStyle(&next_button,0);    
    buttonStyle(&prev_button,0);    
    dispName(50,seqName);
    backpacknint(20, 80, fptr+1,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);    
    backpacknint(84, 80, fcount,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
    backpackcharfont(68,80,'/',BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
    clearBuffers();
    getWavInfo();
    showWavInfo();      //on LCD
    showTimes();
    lasttmr=tmr;
    lastseqtime=seqtime;
    while(1){
        //check if data needs to be buffered                               
        if(WAV_state){
            n=1;            
            while(n==1){
                n=checkWavBuffer();
                //uart_printi(n);
                //uart_print("=checkwav\r\n");
            }
        }
        if(DLC_seq_state){
            n=1;            
            while(n==1){
                n=checkSeqBuffer();
                //uart_printi(n);
                //uart_print("=checkseq\r\n");
            }            
        }
        if((seqtime!=lastseqtime)||(tmr!=lasttmr)){
            showTimes();
            lasttmr=tmr;
            lastseqtime=seqtime;
        }
        
        if((pState==2)&&(DLC_seq_state==0)){        //playback at end of sequence/track
            //clear state and reload
            clearBuffers();
            r=getFile(fptr);
            uart_printi(r);
            uart_print("=getfile\r\n");   
            dispName(50,seqName);
            backpacknint(20, 80, fptr+1,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);    
            backpacknint(84, 80, fcount,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
            getWavInfo();
            showWavInfo();      //on LCD
            pState=0;
            if(loopMode==1){                           //restart if looping one
                n=1;            
                while(n==1){
                    n=checkWavBuffer();
                }
                n=1;            
                while(n==1){
                    n=checkSeqBuffer();
                }            
                pState=2;
                WAV_state=1;
                DLC_seq_state=1;
            }            
            if(loopMode==2){                           //load next and start if looping all
                clearBuffers();                         //reset
                fptr=fptr+1;                            //load next
                if(fptr>=fcount){fptr=0;}
                r=getFile(fptr);
                uart_printi(r);
                uart_print("=getfile\r\n");   
                dispName(50,seqName);
                backpacknint(20, 80, fptr+1,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);    
                backpacknint(84, 80, fcount,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
                getWavInfo();
                showWavInfo();      //on LCD
                n=1;            
                while(n==1){
                    n=checkWavBuffer();
                }
                n=1;            
                while(n==1){
                    n=checkSeqBuffer();
                }            
                pState=2;
                WAV_state=1;
                DLC_seq_state=1;
            }            
        }
        if((pState==2)&&(DLC_seq_state==0)){        //paused but has stopped
            //clear state and reload
            clearBuffers();
            r=getFile(fptr);
            uart_printi(r);
            uart_print("=getfile\r\n");   
            dispName(50,seqName);
            backpacknint(20, 80, fptr+1,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);    
            backpacknint(84, 80, fcount,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
            getWavInfo();
            showWavInfo();      //on LCD
            pState=0;            
        }
        
        if(backpackbuttonobjecttouched(&scan)){
            buttonRelease(&scan);
            pState=0;
            WAV_state=0;
            DLC_seq_state=0;
            clearBuffers();
            break;
        }               
        if(backpackbuttonobjecttouched(&play)){
            buttonRelease(&play);                       
            if(pState==0){
                n=1;            
                while(n==1){
                    n=checkWavBuffer();
                }
                n=1;            
                while(n==1){
                    n=checkSeqBuffer();
                }            
                pState=2;
                WAV_state=1;
                DLC_seq_state=1;
            }
            if(pState==1){      //resume
                pState=2;
                WAV_state=1;
                DLC_seq_state=1;
            }            
        }               
        if(backpackbuttonobjecttouched(&stop)){
            buttonRelease(&stop);
            switch(pState){
                case 0:  break;     //do nothing
                case 1:             //paused > play    
                    pState=2;       //pause
                    WAV_state=1;    //resume wav output
                    DLC_seq_state=1;    //resume seq output
                    break;
                case 2:             //play > paused
                    pState=1;       //pause
                    WAV_state=0;    //pause wav output
                    DLC_seq_state=0;    //pause seq output
                    break;
            }
            buttonStyle(&stop,pauseButtonState);        //redraw at current state
        }
        //only toggle if needed
        if(pauseButtonState==0){
            if(pState==1){
                pauseButtonState=1;
                buttonStyle(&stop,1);    
            }
        }else{
            if(pState!=1){
                pauseButtonState=0;
                buttonStyle(&stop,0);    
            }
        }
        if(backpackbuttonobjecttouched(&play_loop)){
            switch(loopMode){
                case 0: loopMode=1;play_loop.text=text_loop_one;break;
                case 1: loopMode=2;play_loop.text=text_loop_all;break;
                default:loopMode=0;play_loop.text=text_loop_off;break;
            }
            //buttonStyle(&play_loop,0);      //show new
            buttonRelease(&play_loop);
            if(loopMode){buttonStyle(&play_loop,1);}      //show highlighted if looping
        }               
        if(backpackbuttonobjecttouched(&next_button)){
            buttonRelease(&next_button);
            //pState=0;
            WAV_state=0;
            DLC_seq_state=0;
            clearBuffers();
            fptr=fptr+1;
            if(fptr>=fcount){fptr=0;}
            r=getFile(fptr);
            uart_printi(r);
            uart_print("=getfile\r\n");   
            dispName(50,seqName);
            backpacknint(20, 80, fptr+1,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);    
            backpacknint(84, 80, fcount,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
            getWavInfo();
            showWavInfo();      //on LCD
            switch(pState){
                case 0: break;      //was stopped, do nothing
                case 1:             //was paused
                    n=1;            
                    while(n==1){
                        n=checkWavBuffer();
                    }
                    n=1;            
                    while(n==1){
                        n=checkSeqBuffer();
                    }            
                    pState=1;
                    WAV_state=0;
                    DLC_seq_state=0;
                    break;
                case 2:             //was playing
                    n=1;            
                    while(n==1){
                        n=checkWavBuffer();
                    }
                    n=1;            
                    while(n==1){
                        n=checkSeqBuffer();
                    }            
                    pState=2;
                    WAV_state=1;
                    DLC_seq_state=1;
                    break;
                default: pState=0;
            }
        }               
        if(backpackbuttonobjecttouched(&prev_button)){
            buttonRelease(&prev_button);
            //pState=0;
            WAV_state=0;
            DLC_seq_state=0;
            clearBuffers();
            if(pState<2){  //previous track if stopped or paused, otherwise back to start of current
                fptr=fptr-1;
                if(fptr<0){fptr=fptr+fcount;}
                if(fptr<0){fptr=0;}
            }
            r=getFile(fptr);
            uart_printi(r);
            uart_print("=getfile\r\n");   
            dispName(50,seqName);
            backpacknint(20, 80, fptr+1,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);    
            backpacknint(84, 80, fcount,3, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
            getWavInfo();
            showWavInfo();      //on LCD
            switch(pState){
                case 0: break;      //was stopped, do nothing
                case 1:             //was paused
                    n=1;            
                    while(n==1){
                        n=checkWavBuffer();
                    }
                    n=1;            
                    while(n==1){
                        n=checkSeqBuffer();
                    }            
                    pState=1;
                    WAV_state=0;
                    DLC_seq_state=0;
                    break;
                case 2:             //was playing
                    n=1;            
                    while(n==1){
                        n=checkWavBuffer();
                    }
                    n=1;            
                    while(n==1){
                        n=checkSeqBuffer();
                    }            
                    pState=2;
                    WAV_state=1;
                    DLC_seq_state=1;
                    break;
                default: pState=0;
            }
        }               
    }
    for(j=1;j<65;j++){DLC_data[j]=0;}       //turn all off
    scan.text=sOld; //restore button    
}

void showMessage(int i){     //display error message/file state
    int c;
    c=BUTTON_OTHER;
    if(i<=0){c=BUTTON_ERROR;}
    backpacknint(120, 228, abs(i),3, c,BUTTON_BACK,BUTTON_FONT);    
    if(i>=0){backpackchararrayfont(184,228,"Files found",c,BLACK,BUTTON_FONT);}
    else if(i<-3){backpackchararrayfont(184,228,"File error",c,BLACK,BUTTON_FONT);}
    else if(i==-3){backpackchararrayfont(184,228,"Not ready",c,BLACK,BUTTON_FONT);}
    else if(i==-2){backpackchararrayfont(184,228,"Int error",c,BLACK,BUTTON_FONT);}
    else if(i==-1){backpackchararrayfont(184,228,"Disk error",c,BLACK,BUTTON_FONT);}    
}

int getFile(int n){     //load file n by index
    char fname[256]=""; //for manipulating filenames
    char ntemp[256]="";
    char wavname[20]="";
    uart_printi(n);
    uart_print("=n\r\n");            
    int i=0;
    int j;
    fresult=f_opendir(&dd,rootbuf);
    uart_printi(fresult);
    uart_print("=opendir\r\n");        
    while(i<255){
        fresult=f_readdir(&dd,&fno);
        uart_printi(fresult);
        uart_print("=f_readdir\r\n");   
        if((fresult!=FR_OK)||(fno.fname[0]==0)){        //fail or empty name
            return 0;
        }else{
            if(((fno.fattrib&AM_DIR)==0)&&checkifLS(fno.fname)){
                uart_printi(i);
                uart_print(" : ");
                uart_print(fno.fname);
                uart_print(" : ");
                uart_print(fno.lfname);
                uart_print("\r\n");
                uart_printi(strlen(fno.lfname));
                uart_print("=LFLEN\r\n");
                seqName[0]=0;       //trim
                uart_printi(scopy(fno.lfname,ntemp,255));
                uart_print("=scopy\r\n"); 
                if(strlen(ntemp)>28){
                    ntemp[25]='.';
                    ntemp[26]='.';
                    ntemp[27]='.';
                    ntemp[28]=0;
                }
                if(i==n){
                    fresult=f_open(&seqfile,fno.fname,FA_READ);
                    uart_printi(fresult);
                    uart_print("=open seq\r\n");                       
                    fresult=f_lseek(&seqfile,0);        //rewind
                    uart_printi(fresult);
                    uart_print("=fseek\r\n");                    
                    getSeqtime();
                    if(fresult!=FR_OK){return 0;}                                        
                    seqName[0]=0;       //trim
                    uart_printi(scopy(ntemp,seqName,255));
                    uart_print("=scopy\r\n");                       
                    uart_print(seqName);
                    uart_print("=NAME\r\n");                                           
                    uart_print(fno.lfname);
                    uart_print("=LFNAME\r\n");                                           
                    uart_printi(setWavName(wavname,fno.fname));
                    uart_print("=WAVNAME\r\n");                                                               
                    uart_print(wavname);                                                               
                    uart_print("=NAME\r\n");  
                    fresult=f_open(&wavfile,wavname,FA_READ);
                    uart_printi(fresult);
                    uart_print("=open wav\r\n");                       
                    if(fresult!=FR_OK){return 2;}   //wav did not load                                                         
                    return 1;
                }
                i++;
            }
        }

    }    
    return 1;
}

int checkifLS(char* s){
    //if name ends in .lsq or .lsn, return 1
    int t;
    t=strlen(s);
    if(t<4){return 0;}  //name too short
    if(s[t-4] != '.'){return 0;}
    if((s[t-3] != 'L')&&(s[t-3] != 'l')){return 0;}
    if((s[t-2] != 'S')&&(s[t-2] != 's')){return 0;}
    if((s[t-1] != 'Q')&&(s[t-1] != 'q')&&(s[t-1] != 'N')&&(s[t-1] != 'n')){return 0;}
    return 1;    
}

int scopy(char* src, char* dst, int n){
    int i=0;    
    while(i<(n-1)){
        dst[i]=src[i];
        if(dst[i]==0){return i+1;}
        i++;
    }
    dst[i]=0;
    return i;
}

void dispName(int y, char* s){
    backpackbox(0,y,479,y+23,BUTTON_BACK);
    if(s[0]==0){
        backpackchararrayfont(184,y,"NO FILE",BUTTON_ERROR,BUTTON_BACK,BUTTON_FONT);
    }else{
        int x=0;
        x=16*strlen(s);
        if(x>480){x=480;}
        backpackchararrayfont(240-x/2,y,s,BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
    }
}

int setWavName(char* w, char*s){
    int n;
    n=scopy(s,w,14)-1;    
    if(n<4){w[0]=0;return 0;}   //too short to work
    w[n-4]='.';
    w[n-3]='W';
    w[n-2]='A';
    w[n-1]='V';
    return n;
}

void fillWavBuffer(void){
    //does checkWavBuffer() till buffers are full
    char n=1;   
    while(n==1){n=checkWavBuffer();}
}

char checkWavBuffer(void){
    int n;
    if(wavBufferBytes[wavPage]==0){
        fresult=f_read(&wavfile, wavBuffer[wavPage],BUFFERSIZE,&n);
        //uart_printi(fresult);
        //uart_print("=checkwav\r\n");
        if(fresult==FR_OK){
            wavBufferBytes[wavPage]=n;
            wavPage++;
            if(wavPage>=BUFFERCOUNT){wavPage=0;}
            return 1;       //all good, keep reading
        }
    }else{
        return 0;   //no read needed, this flags the while loop to finish
    }
    return -1;      //error, stop scanning
}

char checkSeqBuffer(void){
    int n;
    if(seqBufferBytes[seqPage]==0){
        fresult=f_read(&seqfile, seqBuffer[seqPage],BUFFERSIZE,&n);
        //uart_printi(fresult);
        //uart_print("=checkwav\r\n");
        if(fresult==FR_OK){
            seqBufferBytes[seqPage]=n;
            seqPage++;
            if(seqPage>=BUFFERCOUNT){seqPage=0;}
            return 1;       //all good, keep reading
        }
    }else{
        return 0;   //no read needed, this flags the while loop to finish
    }
    return -1;      //error, stop scanning
}

int getWavInfo(){
    char hdr[44];
    WAVHeader w;
    int n=0;
    int bytesPerSample=0;
    //clear all
    wav_sample_rate=0;
    wav_left_offset=0;
    wav_right_offset=0;
    wav_bytes_per_sample=0;
    fresult=f_lseek(&wavfile,0);        //rewind
    uart_printi(fresult);
    uart_print("=fseek\r\n");
    fresult=f_read(&wavfile, hdr,44,&n);
    uart_printi(fresult);
    uart_print("=readhdr\r\n");
    if(fresult==FR_OK){
        WAVreadWAVHeader(hdr,&w,n);
        uart_printi(w.fmtHeader.sampleRate);
        uart_print("=sr\r\n");
        uart_printi(w.fmtHeader.audioFormat);
        uart_print("=af\r\n");
        uart_printi(w.fmtHeader.numChannels);
        uart_print("=channels\r\n");
        uart_printi(w.fmtHeader.bitsPerSample);
        uart_print("=bits per samp\r\n");
        if(w.fmtHeader.audioFormat!=0x01){return 0;}   //can only play PCM, format code 0x01
        bytesPerSample=w.fmtHeader.bitsPerSample/8;    //per channel
        //transfer to working variables
        wav_bytes_per_sample=bytesPerSample*w.fmtHeader.numChannels;
        if(bytesPerSample==1){
            wav_signed=0;
        }else{
            wav_signed=1;            
        }
        wav_sample_rate=w.fmtHeader.sampleRate;
        wav_left_offset=bytesPerSample-1;
        if(w.fmtHeader.numChannels>1){
            wav_right_offset=wav_left_offset+bytesPerSample;
        }else{
            wav_right_offset=wav_left_offset;
        }
        uart_printi(wav_left_offset);
        uart_print("=left offset\r\n");
        uart_printi(wav_right_offset);
        uart_print("=right offset\r\n");
        if(wav_bytes_per_sample<1){return 0;}       //not valid
        if(wav_sample_rate<1){return 0;}            //not valid
        return 1;
    }    
    return 0;    
}

void showWavInfo(void){
    int c,b;
    if(wav_right_offset==wav_left_offset){
        c=1;
    }else{
        c=2;
    }
    b=(wav_bytes_per_sample/c)*8;
    if((wav_sample_rate>0)&&(wav_bytes_per_sample>0)){
        backpacknint(240,80,wav_sample_rate/1000,2,BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
        backpackchararrayfont(272,80,"kHz ",BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);        
        backpacknint(336,80,b,2,BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
        backpackchararrayfont(368,80,"B ",BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);        
        if(c==1){
            backpackcharfont(400,80,'M',BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);                    
        }else{
            backpackcharfont(400,80,'S',BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);        
        }
    }else{
        backpackchararrayfont(240,80,"NO WAV DATA",BUTTON_ERROR,BUTTON_BACK,BUTTON_FONT);        
    }
}

//move this here  so that it can call back while busy
void buttonRelease(struct backpackbutton *b){
    fillWavBuffer();
    buttonStyle(b,1);                           //draw pressed
    while(backpackbuttonobjecttouched(b)){     //wait til release
        fillWavBuffer();
    }
    buttonStyle(b,0);                           //draw released
    fillWavBuffer();
    delay(1);                                  //wait a bit for debounce    
    fillWavBuffer();
}

void getSeqtime(){
    int n=2;
    char d[2];    
    seqtime=0;
    fresult=f_lseek(&seqfile,0);        //rewind
    uart_printi(fresult);
    uart_print("=fseek\r\n");
    while(n==2){        
        fresult=f_read(&seqfile, d,2,&n);    //read
        seqtime=seqtime+parse_word_no_action(d[0],d[1]);
    }
    uart_printi(seqtime);
    uart_print("=seqtime\r\n");
    fresult=f_lseek(&seqfile,0);        //rewind
    uart_printi(fresult);
    uart_print("=fseek\r\n");
}

void showTimes(void){
    backpackshowms(200,5,tmr/1000, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
    backpackcharfont(296,5,'/',BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);        
    backpackshowms(312,5,seqtime/1000, BUTTON_FORE,BUTTON_BACK,BUTTON_FONT);
}

void clearBuffers(void){
    int j;
    //shut down
    DLC_seq_state=0;
    WAV_state=0;
    //clear
    seqPage=0;
    wavPage=0;
    for(j=0;j<BUFFERCOUNT;j++){
        wavBufferBytes[j]=0;
        seqBufferBytes[j]=0;        
    }
    wavBufferPtr=0;
    seqBufferPtr=0;
    wavBufferInUse=0;
    seqBufferInUse=0;
    tmr=0;            //timer counter incremented by DLC ISR
    seqtmr=0;
    DLC_data[0]=0;
    for(j=0;j<LAMPS;j++){
        DLC_data[j+1]=0;
        rampTime[j]=0;
        rampSteps[j]=0;
    }
}